<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Sale;
use App\Models\SaleItem;
use App\Models\Product;
use App\Models\Category;
use App\Models\User;
use Carbon\Carbon; 
use Illuminate\Support\Facades\Log;

class SaleController extends Controller
{
    public function index(Request $request)
    {
        $query = Product::with('category');

        if ($request->has('q') && !empty($request->q)) {
            $query->where('product_name', 'like', '%' . $request->q . '%');
        }

        $products = $query->get();
        $categories = Category::with('products')->get();
        // ✅ only active waiters
        $waiters = User::where('is_role', 3)->where('status', 1)->get();

        return view('sales.list', compact('products', 'categories', 'waiters'));
    }

    public function addToCart(Request $request)
    {
        $product = Product::findOrFail($request->product_id);

        $cart = session()->get('cart', []);

        if (isset($cart[$product->id])) {
            $cart[$product->id]['quantity']++;
        } else {
            $cart[$product->id] = [
                'id' => $product->id,
                'name' => $product->product_name,
                'selling_price' => $product->selling_price,
                'quantity' => 1
            ];
        }

        session()->put('cart', $cart);

        return response()->json([
            'success' => true,
            'cart' => $cart
        ]);
    }

    public function updateCart(Request $request)
    {
        $cart = session()->get('cart', []);

        if (isset($cart[$request->product_id])) {
            if ($request->action === 'increase') {
                $cart[$request->product_id]['quantity']++;
            } elseif ($request->action === 'decrease') {
                if ($cart[$request->product_id]['quantity'] > 1) {
                    $cart[$request->product_id]['quantity']--;
                } else {
                    unset($cart[$request->product_id]);
                }
            }

            session()->put('cart', $cart);
        }

        return response()->json([
            'success' => true,
            'cart' => $cart
        ]);
    }

    public function removeFromCart(Request $request)
    {
        $cart = session()->get('cart', []);

        if (isset($cart[$request->product_id])) {
            unset($cart[$request->product_id]);
            session()->put('cart', $cart);
        }

        return response()->json([
            'success' => true,
            'cart' => $cart
        ]);
    }

    public function clearCart()
    {
        session()->forget('cart');

        return response()->json([
            'success' => true,
            'cart' => []
        ]);
    }

    public function store(Request $request)
    {
        try {
            $request->validate([
                'waiter_id' => 'required|exists:users,id',
                'sale_type' => 'required|in:pay,bill',
                'cart_items' => 'required|array|min:1'
            ]);

            $cartItems = $request->cart_items;
            if (empty($cartItems)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cart is empty'
                ]);
            }

            // Calculate totals
            $total = collect($cartItems)->sum(fn($item) => $item['selling_price'] * $item['quantity']);
            $subtotal = $total * 0.82;
            $tax = $total * 0.18;

            $status = 'pending';
            $cashier_id = null;
            $verified_at = null;

            if ($request->sale_type === 'pay') {
                $status = 'verified';
                $cashier_id = auth()->id();
                $verified_at = Carbon::now();
            }

            // 🧠 DUPLICATE CHECK (within 2 minutes)
            $recentDuplicate = Sale::where('waiter_id', $request->waiter_id)
                ->whereBetween('created_at', [Carbon::now()->subMinutes(2), Carbon::now()])
                ->where('total', $total)
                ->whereHas('items', function ($query) use ($cartItems) {
                    $productIds = collect($cartItems)->pluck('id')->toArray();
                    $query->whereIn('product_id', $productIds);
                })
                ->latest()
                ->first();

            if ($recentDuplicate) {
                $minutesAgo = Carbon::parse($recentDuplicate->created_at)->diffInMinutes(Carbon::now());
                $timeRecorded = Carbon::parse($recentDuplicate->created_at)->format('H:i');

                return response()->json([
                    'success' => false,
                    'message' => "DUPLICATE TRANSACTION DETECTED: A similar sale was already recorded {$minutesAgo} minute(s) ago at {$timeRecorded}. check your DASHBOARD"
                ], 409);
            }

            // 🧾 Create the sale
            $sale = Sale::create([
                'waiter_id' => $request->waiter_id,
                'cashier_id' => $cashier_id,
                'payment_type' => $request->payment_type,
                'subtotal' => $subtotal,
                'tax' => $tax,
                'total' => $total,
                'status' => $status,
                'verified_at' => $verified_at,
            ]);

            // 🔹 Save items securely
            foreach ($cartItems as $item) {
                $product = Product::find($item['id']);
                if (!$product) continue;

                SaleItem::create([
                    'sale_id' => $sale->id,
                    'product_id' => $product->id,
                    'quantity' => $item['quantity'],
                    'price' => $product->selling_price,
                    'total' => $product->selling_price * $item['quantity']
                ]);
            }

            return response()->json([
                'success' => true,
                'sale_id' => $sale->id,
                'cart' => []
            ]);
        } catch (\Exception $e) {
            Log::error('Sale creation error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error processing sale: ' . $e->getMessage()
            ], 500);
        }
    }

    public function showReceipt($id)
    {
        $sale = Sale::with(['items.product', 'waiter', 'cashier'])->findOrFail($id);
        return view('partials.receipt', compact('sale'));
    }

    public function categoryProducts($id)
    {
        $category = Category::with('products')->findOrFail($id);
        return response()->json($category->products);
    }

    public function search(Request $request)
    {
        $query = Product::with('category');

        if ($request->has('q') && !empty($request->q)) {
            $query->where('product_name', 'like', '%' . $request->q . '%');
        }

        $products = $query->get();
        $categories = Category::with('products')->get();
        // ✅ only active waiters
        $waiters = User::where('is_role', 3)->where('status', 1)->get();

        return view('sales.list', compact('products', 'categories', 'waiters'));
    }
}