<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Sale;
use App\Models\SaleItem;
use App\Models\User;
use App\Models\Product;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\SalesExport;
use Carbon\Carbon; 
use PDF;
use Illuminate\Support\Facades\DB;

class SaleReportController extends Controller
{
    public function report(Request $request)
    {
        $currentUser = auth()->user();

        // Set waiters and cashiers based on user role
        if ($currentUser->is_role == 1) { // Admin
            $waiters = User::where('is_role', 3)->get();
            $cashiers = User::whereIn('is_role', [1, 2])->get();
        } elseif ($currentUser->is_role == 2) { // Cashier
            $waiters = User::where('is_role', 3)->get();
            $cashiers = User::where('id', $currentUser->id)->get();
        } else { // Waiter
            $waiters = User::where('id', $currentUser->id)->get();
            $cashiers = collect();
        }

        // Get active tab
        $activeTab = $request->get('tab', 'detailed');
        $dateRange = $request->filled('date_range') ? $request->date_range : 'today';

        // Load ALL data for ALL tabs
        $allData = $this->getAllReportData($request);

        return view('sales.report', array_merge([
            'waiters' => $waiters,
            'cashiers' => $cashiers,
            'activeTab' => $activeTab,
            'dateRange' => $dateRange,
        ], $allData));
    }

    /**
     * Get ALL report data for ALL tabs
     */
    private function getAllReportData($request)
    {
        return [
            // Detailed Sales
            'detailedSales' => $this->getDetailedData($request),
            'detailedTotals' => $this->getDetailedTotals($request),
            
            // Daily Summary
            'dailySales' => $this->getDailyData($request),
            'dailyTotals' => $this->getDailyTotals($request),
            
            // Weekly Summary
            'weeklySales' => $this->getWeeklyData($request),
            'weeklyTotals' => $this->getWeeklyTotals($request),
            
            // Monthly Summary
            'monthlySales' => $this->getMonthlyData($request),
            'monthlyTotals' => $this->getMonthlyTotals($request),
            
            // Product Performance
            'productPerformance' => $this->getProductsData($request),
            'productTotals' => $this->getProductsTotals($request),
            
            // Waiter Performance
            'waiterPerformance' => $this->getWaitersData($request),
            'waiterTotals' => $this->getWaitersTotals($request),
            
            // Hourly Analysis
            'hourlySales' => $this->getHourlyData($request),
            'hourlyTotals' => $this->getHourlyTotals($request),
            
            // Summary Statistics
            'summaryStats' => $this->getSummaryStats($request),
        ];
    }

    /**
     * Detailed Sales Data
     */
    private function getDetailedData($request)
    {
        $query = Sale::with(['waiter', 'cashier']);
        $this->applyFilters($query, $request);
        return $query->orderBy('created_at', 'desc')->paginate(20);
    }

    private function getDetailedTotals($request)
    {
        $query = Sale::query();
        $this->applyFilters($query, $request);
        
        $sales = $query->get();
        
        return [
            'subtotal' => $sales->sum(function($sale) {
                return floatval($sale->subtotal);
            }),
            'tax' => $sales->sum(function($sale) {
                return floatval($sale->tax);
            }),
            'total' => $sales->sum('total'),
            'count' => $sales->count(),
        ];
    }

    /**
     * Daily Sales Data
     */
    private function getDailyData($request)
    {
        $query = Sale::select(
                DB::raw('DATE(created_at) as date'),
                DB::raw('COUNT(*) as total_orders'),
                DB::raw('SUM(CAST(subtotal AS DECIMAL(12,2))) as total_subtotal'),
                DB::raw('SUM(CAST(tax AS DECIMAL(12,2))) as total_tax'),
                DB::raw('SUM(total) as total_amount'),
                DB::raw('AVG(total) as average_order_value'),
                DB::raw('DAYNAME(created_at) as day_name')
            )
            ->groupBy(DB::raw('DATE(created_at)'), DB::raw('DAYNAME(created_at)'))
            ->orderBy('date', 'desc');

        $this->applyFilters($query, $request);
        return $query->paginate(15);
    }

    private function getDailyTotals($request)
    {
        $query = Sale::select(
                DB::raw('COUNT(*) as total_orders'),
                DB::raw('SUM(total) as total_amount'),
                DB::raw('SUM(CAST(subtotal AS DECIMAL(12,2))) as total_subtotal'),
                DB::raw('SUM(CAST(tax AS DECIMAL(12,2))) as total_tax')
            );

        $this->applyFilters($query, $request);
        $totals = $query->first();

        return [
            'total_orders' => $totals->total_orders ?? 0,
            'total_amount' => $totals->total_amount ?? 0,
            'total_subtotal' => $totals->total_subtotal ?? 0,
            'total_tax' => $totals->total_tax ?? 0,
        ];
    }

    /**
     * Weekly Summary Data
     */
    private function getWeeklyData($request)
    {
        $query = Sale::select(
                DB::raw('YEAR(created_at) as year'),
                DB::raw('WEEK(created_at, 1) as week_number'),
                DB::raw('MIN(DATE(created_at)) as week_start'),
                DB::raw('MAX(DATE(created_at)) as week_end'),
                DB::raw('COUNT(*) as total_orders'),
                DB::raw('SUM(total) as total_amount'),
                DB::raw('SUM(CAST(subtotal AS DECIMAL(12,2))) as total_subtotal'),
                DB::raw('SUM(CAST(tax AS DECIMAL(12,2))) as total_tax'),
                DB::raw('AVG(total) as average_order_value')
            )
            ->groupBy('year', 'week_number')
            ->orderBy('year', 'desc')
            ->orderBy('week_number', 'desc');

        $this->applyFilters($query, $request);
        return $query->paginate(12);
    }

    private function getWeeklyTotals($request)
    {
        $query = Sale::select(
                DB::raw('COUNT(*) as total_orders'),
                DB::raw('SUM(total) as total_amount')
            );

        $this->applyFilters($query, $request);
        $totals = $query->first();

        return [
            'total_orders' => $totals->total_orders ?? 0,
            'total_amount' => $totals->total_amount ?? 0,
        ];
    }

    /**
     * Monthly Summary Data
     */
    private function getMonthlyData($request)
    {
        $query = Sale::select(
                DB::raw('YEAR(created_at) as year'),
                DB::raw('MONTH(created_at) as month_number'),
                DB::raw('MONTHNAME(created_at) as month_name'),
                DB::raw('COUNT(*) as total_orders'),
                DB::raw('SUM(total) as total_amount'),
                DB::raw('SUM(CAST(subtotal AS DECIMAL(12,2))) as total_subtotal'),
                DB::raw('SUM(CAST(tax AS DECIMAL(12,2))) as total_tax'),
                DB::raw('AVG(total) as average_order_value')
            )
            ->groupBy('year', 'month_number', 'month_name')
            ->orderBy('year', 'desc')
            ->orderBy('month_number', 'desc');

        $this->applyFilters($query, $request);
        $monthlySales = $query->paginate(12);
        
        // Calculate growth for each month
        foreach ($monthlySales as $sale) {
            $previousMonth = Sale::whereYear('created_at', $sale->year)
                ->whereMonth('created_at', $sale->month_number - 1)
                ->sum('total');
            
            if ($previousMonth > 0) {
                $sale->growth = (($sale->total_amount - $previousMonth) / $previousMonth) * 100;
            } else {
                $sale->growth = null;
            }
        }

        return $monthlySales;
    }

    private function getMonthlyTotals($request)
    {
        $query = Sale::select(
                DB::raw('COUNT(*) as total_orders'),
                DB::raw('SUM(total) as total_amount')
            );

        $this->applyFilters($query, $request);
        $totals = $query->first();

        return [
            'total_orders' => $totals->total_orders ?? 0,
            'total_amount' => $totals->total_amount ?? 0,
        ];
    }

    /**
     * Product Performance Data
     */
    private function getProductsData($request)
    {
        $query = SaleItem::select(
                'products.id',
                'products.product_name',
                'categories.category_name',
                DB::raw('SUM(sale_items.quantity) as total_quantity_sold'),
                DB::raw('SUM(sale_items.quantity * sale_items.price) as total_revenue'),
                DB::raw('AVG(sale_items.price) as average_price'),
                DB::raw('COUNT(DISTINCT sale_items.sale_id) as times_ordered')
            )
            ->join('products', 'sale_items.product_id', '=', 'products.id')
            ->leftJoin('categories', 'products.category_id', '=', 'categories.id')
            ->join('sales', 'sale_items.sale_id', '=', 'sales.id')
            ->groupBy('products.id', 'products.product_name', 'categories.category_name')
            ->orderBy('total_revenue', 'desc');

        // Apply date filters through sales table
        if ($request->filled('from') && $request->filled('to')) {
            $query->whereBetween('sales.created_at', [
                $request->from . ' 00:00:00',
                $request->to . ' 23:59:59'
            ]);
        }

        if ($request->filled('waiter_id')) {
            $query->where('sales.waiter_id', $request->waiter_id);
        }

        $products = $query->paginate(20);
        
        // Calculate revenue per item
        foreach ($products as $product) {
            $product->revenue_per_item = $product->total_quantity_sold > 0 ? 
                $product->total_revenue / $product->total_quantity_sold : 0;
        }

        return $products;
    }

    private function getProductsTotals($request)
    {
        $query = SaleItem::select(
                DB::raw('SUM(sale_items.quantity) as total_quantity'),
                DB::raw('SUM(sale_items.quantity * sale_items.price) as total_revenue')
            )
            ->join('sales', 'sale_items.sale_id', '=', 'sales.id');

        // Apply date filters through sales table
        if ($request->filled('from') && $request->filled('to')) {
            $query->whereBetween('sales.created_at', [
                $request->from . ' 00:00:00',
                $request->to . ' 23:59:59'
            ]);
        }

        if ($request->filled('waiter_id')) {
            $query->where('sales.waiter_id', $request->waiter_id);
        }

        $totals = $query->first();

        return [
            'total_quantity' => $totals->total_quantity ?? 0,
            'total_revenue' => $totals->total_revenue ?? 0,
        ];
    }

    /**
     * Waiter Performance Data
     */
    private function getWaitersData($request)
    {
        $query = Sale::select(
                'users.id',
                'users.name as waiter_name',
                DB::raw('COUNT(*) as total_orders'),
                DB::raw('SUM(total) as total_sales'),
                DB::raw('SUM(CAST(subtotal AS DECIMAL(12,2))) as total_subtotal'),
                DB::raw('SUM(CAST(tax AS DECIMAL(12,2))) as total_tax'),
                DB::raw('AVG(total) as average_order_value'),
                DB::raw('MAX(total) as highest_sale'),
                DB::raw('MIN(total) as lowest_sale')
            )
            ->join('users', 'sales.waiter_id', '=', 'users.id')
            ->where('users.is_role', 3)
            ->groupBy('users.id', 'users.name')
            ->orderBy('total_sales', 'desc');

        $this->applyFilters($query, $request);

        $waiters = $query->paginate(15);
        
        // Calculate performance rating
        foreach ($waiters as $waiter) {
            $avgOrder = $waiter->average_order_value;
            if ($avgOrder > 50000) {
                $waiter->performance_rating = 'Excellent';
                $waiter->performance_class = 'success';
            } elseif ($avgOrder > 25000) {
                $waiter->performance_rating = 'Good';
                $waiter->performance_class = 'info';
            } elseif ($avgOrder > 10000) {
                $waiter->performance_rating = 'Average';
                $waiter->performance_class = 'warning';
            } else {
                $waiter->performance_rating = 'Needs Improvement';
                $waiter->performance_class = 'danger';
            }
        }

        return $waiters;
    }

    private function getWaitersTotals($request)
    {
        $query = Sale::select(
                DB::raw('COUNT(*) as total_orders'),
                DB::raw('SUM(total) as total_sales')
            )
            ->join('users', 'sales.waiter_id', '=', 'users.id')
            ->where('users.is_role', 3);

        $this->applyFilters($query, $request);
        $totals = $query->first();

        return [
            'total_orders' => $totals->total_orders ?? 0,
            'total_sales' => $totals->total_sales ?? 0,
        ];
    }

    /**
     * Hourly Sales Data
     */
    private function getHourlyData($request)
    {
        $query = Sale::select(
                DB::raw('HOUR(created_at) as hour'),
                DB::raw('COUNT(*) as total_orders'),
                DB::raw('SUM(total) as total_amount'),
                DB::raw('AVG(total) as average_order_value')
            )
            ->groupBy('hour')
            ->orderBy('hour');

        $this->applyFilters($query, $request);

        $hourlyData = $query->paginate(24);
        
        // Add formatted time periods
        foreach ($hourlyData as $data) {
            $data->hour_formatted = sprintf('%02d:00 - %02d:00', $data->hour, $data->hour + 1);
            
            if ($data->hour >= 6 && $data->hour < 12) {
                $data->time_period = 'Morning';
                $data->period_class = 'info';
            } elseif ($data->hour >= 12 && $data->hour < 17) {
                $data->time_period = 'Afternoon';
                $data->period_class = 'success';
            } elseif ($data->hour >= 17 && $data->hour < 22) {
                $data->time_period = 'Evening';
                $data->period_class = 'warning';
            } else {
                $data->time_period = 'Night';
                $data->period_class = 'dark';
            }
        }

        return $hourlyData;
    }

    private function getHourlyTotals($request)
    {
        $query = Sale::select(
                DB::raw('COUNT(*) as total_orders'),
                DB::raw('SUM(total) as total_amount')
            );

        $this->applyFilters($query, $request);
        $totals = $query->first();

        return [
            'total_orders' => $totals->total_orders ?? 0,
            'total_amount' => $totals->total_amount ?? 0,
        ];
    }

    /**
     * Summary Statistics
     */
    private function getSummaryStats($request)
    {
        $query = Sale::query();
        $this->applyFilters($query, $request);

        $sales = $query->get();

        return [
            'total_sales' => $sales->sum('total') ?? 0,
            'total_orders' => $sales->count(),
            'average_order_value' => $sales->avg('total') ?? 0,
            'total_tax' => $sales->sum(function($sale) {
                return floatval($sale->tax);
            }),
            'pending_orders' => $sales->where('status', 'pending')->count(),
            'verified_orders' => $sales->where('status', 'verified')->count(),
        ];
    }

    /**
     * Common filter application method
     */
    private function applyFilters($query, $request)
    {
        // Apply DATE FILTERS
        if ($request->filled('from') && $request->filled('to')) {
            $query->whereBetween('created_at', [
                $request->from . ' 00:00:00',
                $request->to . ' 23:59:59'
            ]);
        } elseif ($request->filled('date_range')) {
            $this->applyDateRangeFilter($query, $request->date_range);
        }

        // Apply other filters
        if ($request->filled('waiter_id')) {
            $query->where('waiter_id', $request->waiter_id);
        }

        if ($request->filled('cashier_id')) {
            $query->where('cashier_id', $request->cashier_id);
        }

        if ($request->filled('payment_type')) {
            $query->where('payment_type', $request->payment_type);
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }
    }

    /**
     * Apply predefined date ranges
     */
    private function applyDateRangeFilter($query, $dateRange)
    {
        $today = Carbon::today();
        
        switch ($dateRange) {
            case 'today':
                $query->whereDate('created_at', $today);
                break;
            case 'yesterday':
                $query->whereDate('created_at', $today->copy()->subDay());
                break;
            case 'this_week':
                $query->whereBetween('created_at', [
                    $today->copy()->startOfWeek(),
                    $today->copy()->endOfWeek()
                ]);
                break;
            case 'last_week':
                $query->whereBetween('created_at', [
                    $today->copy()->subWeek()->startOfWeek(),
                    $today->copy()->subWeek()->endOfWeek()
                ]);
                break;
            case 'this_month':
                $query->whereBetween('created_at', [
                    $today->copy()->startOfMonth(),
                    $today->copy()->endOfMonth()
                ]);
                break;
            case 'last_month':
                $query->whereBetween('created_at', [
                    $today->copy()->subMonth()->startOfMonth(),
                    $today->copy()->subMonth()->endOfMonth()
                ]);
                break;
            case 'last_3_months':
                $query->whereBetween('created_at', [
                    $today->copy()->subMonths(3)->startOfDay(),
                    $today->copy()->endOfDay()
                ]);
                break;
            case 'this_year':
                $query->whereYear('created_at', $today->year);
                break;
        }
    }

    // Export methods remain the same...
    public function exportExcel(Request $request)
    {
        $filters = $request->all();
        $filename = $this->generateExportFilename($filters, 'xlsx');
        
        return Excel::download(new SalesExport($request), $filename);
    }

    public function exportCSV(Request $request)
    {
        $filters = $request->all();
        $filename = $this->generateExportFilename($filters, 'csv');
        
        return Excel::download(new SalesExport($request), $filename);
    }

    public function exportPDF(Request $request)
    {
        $query = Sale::with(['waiter', 'cashier']);
        $this->applyFilters($query, $request);

        $sales = $query->get();
        $filters = $request->all();
        $filename = $this->generateExportFilename($filters, 'pdf');

        // Calculate totals with varchar conversion
        $totals = [
            'subtotal' => $sales->sum(function($sale) {
                return floatval($sale->subtotal);
            }),
            'tax' => $sales->sum(function($sale) {
                return floatval($sale->tax);
            }),
            'total' => $sales->sum('total'),
        ];

        $pdf = PDF::loadView('sales.pdf', compact('sales', 'totals', 'filters'));
        return $pdf->download($filename);
    }

    private function generateExportFilename($filters, $extension)
    {
        $filename = 'sales_report_';
        
        if (!empty($filters['from']) && !empty($filters['to'])) {
            $fromDate = Carbon::parse($filters['from'])->format('d_M_Y');
            $toDate = Carbon::parse($filters['to'])->format('d_M_Y');
            $filename .= $fromDate . '_to_' . $toDate . '_';
        }
        
        if (!empty($filters['waiter_id'])) {
            $waiter = User::find($filters['waiter_id']);
            $filename .= 'waiter_' . ($waiter ? str_replace(' ', '_', strtolower($waiter->name)) : '') . '_';
        }
        
        if (!empty($filters['cashier_id'])) {
            $cashier = User::find($filters['cashier_id']);
            $filename .= 'cashier_' . ($cashier ? str_replace(' ', '_', strtolower($cashier->name)) : '') . '_';
        }
        
        if (!empty($filters['payment_type'])) {
            $filename .= strtolower($filters['payment_type']) . '_payment_';
        }
        
        if (!empty($filters['status'])) {
            $filename .= strtolower($filters['status']) . '_';
        }
        
        $filename .= 'exported_' . date('Y_m_d_H_i') . '.' . $extension;
        
        return $filename;
    }
}