<?php

namespace App\Http\Controllers;

use App\Models\Stocktaking;
use App\Models\Product;
use App\Models\SaleItem;
use Illuminate\Http\Request;
use Auth;
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;

class StocktakingController extends Controller
{
    public function index()
    {
        return view('stocktaking.list');
    }

    public function getStocktakingData(Request $request)
{
    try {
        $date = $request->get('date', Carbon::today()->format('Y-m-d'));

        Log::info("Loading stocktaking data for date: " . $date);

        $startDate = Carbon::parse($date)->startOfDay();
        $endDate = Carbon::parse($date)->endOfDay();

        // Get all products under Breakfast category
        $products = Product::join('categories', 'products.category_id', '=', 'categories.id')
            ->where('categories.category_name', 'Breakfast')
            ->select('products.*')
            ->get();

        if ($products->isEmpty()) {
            return response()->json([]);
        }

        $stocktakingData = [];

        foreach ($products as $product) {

            // Get or create stock record
            $stockRecord = Stocktaking::firstOrCreate(
                [
                    'product_id' => $product->id,
                    'date' => $date
                ],
                [
                    'opening_stock' => 0,
                    'additional_stock' => 0,
                    'buying_price' => $product->purchase_price,
                    'selling_price' => $product->selling_price,
                    'sold_quantity' => 0,
                    'created_by' => Auth::id()
                ]
            );

            // ✅ NEW: AUTO-CALCULATE SOLD ITEMS FROM VERIFIED SALES
            $soldToday = DB::table('sale_items as si')
                ->join('sales as s', 'si.sale_id', '=', 's.id')
                ->where('si.product_id', $product->id)
                ->whereBetween('s.created_at', [$startDate, $endDate])
                ->sum('si.quantity');

            // ✅ override manual entered sold quantity
            $stockRecord->sold_quantity = $soldToday;

            // Re-calculate amounts using REAL sold quantity
            $amountObtained = $soldToday * $stockRecord->selling_price;
            $profit = $amountObtained - ($soldToday * $stockRecord->buying_price);
            $closedStock = ($stockRecord->opening_stock + $stockRecord->additional_stock) - $soldToday;

            $stocktakingData[] = [
                'id' => $stockRecord->id,
                'product_id' => $product->id,
                'product_name' => $product->product_name,
                'opening_stock' => (int)$stockRecord->opening_stock,
                'additional_stock' => (int)$stockRecord->additional_stock,
                'buying_price' => (float)$stockRecord->buying_price,
                'selling_price' => (float)$stockRecord->selling_price,
                'sold_quantity' => (int)$soldToday,
                'amount_obtained' => (float)$amountObtained,
                'profit' => (float)$profit,
                'closed_stock' => (int)$closedStock
            ];
        }

        Log::info("Successfully loaded " . count($stocktakingData) . " stocktaking records");

        return response()->json($stocktakingData);

    } catch (\Exception $e) {
        Log::error("Error in getStocktakingData: " . $e->getMessage());
        Log::error($e->getTraceAsString());

        return response()->json([
            'error' => 'Server error loading stocktaking data'
        ], 500);
    }
}

    public function saveStocktaking(Request $request)
    {
        $request->validate([
            'stocktaking_data' => 'required|array',
            'date' => 'required|date'
        ]);

        DB::beginTransaction();

        try {
            $date = $request->date;
            $stocktakingData = $request->stocktaking_data;

            foreach ($stocktakingData as $data) {
                Stocktaking::updateOrCreate(
                    [
                        'product_id' => $data['product_id'],
                        'date' => $date
                    ],
                    [
                        'opening_stock' => $data['opening_stock'] ?? 0,
                        'additional_stock' => $data['additional_stock'] ?? 0,
                        'buying_price' => $data['buying_price'] ?? 0,
                        'selling_price' => $data['selling_price'] ?? 0,
                        'sold_quantity' => $data['sold_quantity'] ?? 0,
                        'created_by' => Auth::id()
                    ]
                );
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Stocktaking data saved successfully!'
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error("Error in saveStocktaking: " . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Error saving data'
            ], 500);
        }
    }

    public function resetDailyOpeningStock()
    {
        DB::beginTransaction();

        try {
            $today = Carbon::today();
            $yesterday = Carbon::yesterday();
            
            // Get yesterday's closed stock
            $yesterdayData = Stocktaking::whereDate('date', $yesterday)
                ->get()
                ->keyBy('product_id');

            $products = Product::join('categories', 'products.category_id', '=', 'categories.id')
                ->where('categories.category_name', 'Breakfast')
                ->select('products.*')
                ->get();


            foreach ($products as $product) {
                $yesterdayRecord = $yesterdayData[$product->id] ?? null;
                $openingStock = $yesterdayRecord ? 
                    max(0, ($yesterdayRecord->opening_stock + $yesterdayRecord->additional_stock - $yesterdayRecord->sold_quantity)) : 0;

                Stocktaking::updateOrCreate(
                    [
                        'product_id' => $product->id,
                        'date' => $today
                    ],
                    [
                        'opening_stock' => $openingStock,
                        'additional_stock' => 0,
                        'buying_price' => $product->purchase_price,
                        'selling_price' => $product->selling_price,
                        'sold_quantity' => 0,
                        'created_by' => Auth::id()
                    ]
                );
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Opening stock reset successfully!'
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error("Error in resetDailyOpeningStock: " . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Error resetting stock'
            ], 500);
        }
    }
}