<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\User;
use Illuminate\Support\Facades\Hash;

class UserController extends Controller
{
    // List users (cashier & waiter)
    public function index(Request $request)
    {
        $users = User::whereIn('is_role', [2,3,6])->get();

        // If AJAX request, return JSON
        if ($request->ajax()) {
            return response()->json($users);
        }

        return view('users.index'); // the Blade view
    }

    // Create new user
public function store(Request $request)
{
    $request->validate([
        'name' => 'required|string|max:255',
        'email' => 'required|email|unique:users',
        'password' => 'required|string|min:2',
        'is_role' => 'required|in:1,2,3,6',
        'status' => 'required|in:0,1',
    ]);

    // Extract first name
    $firstName = explode(' ', trim($request->name))[0];

    $user = User::create([
        'name' => $request->name,
        'username' => $firstName,
        'email' => $request->email,
        'password' => Hash::make($request->password),
        'is_role' => $request->is_role,
        'status' => $request->status,
    ]);

    return response()->json([
    'success' => true,
    'user' => $user
]);

}


    // Show form to edit user
    public function edit($id)
    {
        $user = User::findOrFail($id);
        return response()->json($user);
    }

    // Update user info
public function update(Request $request, $id)
{
    $user = User::findOrFail($id);

    $request->validate([
        'name' => 'required|string|max:255',
        'email' => 'required|email|unique:users,email,' . $id,
        'password' => 'nullable|string|min:2',
        'is_role' => 'required|in:1,2,3,6',
        'status' => 'required|in:0,1',
    ]);

    // Update fields
    $user->name = $request->name;
    $user->username = explode(' ', trim($request->name))[0]; // auto update username
    $user->email = $request->email;
    $user->is_role = $request->is_role;
    $user->status = $request->status;

    if ($request->password) {
        $user->password = Hash::make($request->password);
    }

    $user->save();

    return response()->json(['success' => true, 'message' => 'User updated successfully']);
}


    // Delete user
    public function destroy($id)
    {
        $user = User::findOrFail($id);
        $user->delete();

        return response()->json(['success' => true, 'message' => 'User deleted successfully']);
    }
}
