<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Sale;
use Auth;
use Carbon\Carbon;

class ValidatorDashboardController extends Controller
{
    public function dashboard(Request $request)
    {
        $user = Auth::user();

        // Validator only access (role 6)
        if ($user->is_role != 6) {
            abort(403, 'Unauthorized access');
        }

        try {
            // Get all unvalidated sales
            $sales = Sale::where('validity', 'unvalidated')
                        ->with(['waiter', 'cashier', 'items.product'])
                        ->latest()
                        ->get();

            // Calculate stats
            $todayReceiptsCount = Sale::whereDate('created_at', Carbon::today())->count();
            $totalUnvalidatedAmount = Sale::where('validity', 'unvalidated')->sum('total');

            return view('dashboard.validator_list', compact(
                'user',
                'sales',
                'todayReceiptsCount',
                'totalUnvalidatedAmount'
            ));

        } catch (\Exception $e) {
            \Log::error('Validator Dashboard error: ' . $e->getMessage());

            return view('dashboard.validator_list', [
                'user' => Auth::user(),
                'sales' => collect([]),
                'todayReceiptsCount' => 0,
                'totalUnvalidatedAmount' => 0,
            ]);
        }
    }

    public function markAsAttended(Request $request, $saleId)
    {
        $user = Auth::user();

        // Validator only access
        if ($user->is_role != 6) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        try {
            $sale = Sale::where('id', $saleId)
                       ->where('validity', 'unvalidated') // Only update if currently unvalidated
                       ->firstOrFail();
            
            // Mark as validated - overwrite the validity field
            $sale->update([
                'validity' => 'validated',
                'verified_by' => $user->id,
                'verified_at' => Carbon::now(), // Add timestamp for when it was validated
            ]);

            \Log::info("Sale ID {$saleId} validated by user ID {$user->id}");

            return response()->json([
                'success' => true,
                'message' => 'Receipt has been validated successfully'
            ]);

        } catch (\Exception $e) {
            \Log::error('Mark as attended error: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'error' => 'Failed to validate receipt: ' . $e->getMessage()
            ], 500);
        }
    }

    public function getDashboardData(Request $request)
    {
        $user = Auth::user();
        
        if ($user->is_role != 6) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        try {
            $unattendedCount = Sale::where('validity', 'unvalidated')->count();
            $todayReceiptsCount = Sale::whereDate('created_at', Carbon::today())->count();
            $totalUnvalidatedAmount = Sale::where('validity', 'unvalidated')->sum('total');
            
            // Get recent unvalidated sales (last 10 for performance)
            $recentSales = Sale::where('validity', 'unvalidated')
                              ->with(['waiter', 'items.product'])
                              ->latest()
                              ->limit(10)
                              ->get()
                              ->map(function($sale) {
                                  return [
                                      'id' => $sale->id,
                                      'waiter' => $sale->waiter,
                                      'payment_type' => $sale->payment_type,
                                      'subtotal' => $sale->subtotal,
                                      'tax' => $sale->tax,
                                      'total' => $sale->total,
                                      'created_at' => $sale->created_at,
                                      'items' => $sale->items->map(function($item) {
                                          return [
                                              'product' => [
                                                  'product_name' => $item->product->product_name,
                                                  'selling_price' => $item->product->selling_price
                                              ],
                                              'quantity' => $item->quantity
                                          ];
                                      })
                                  ];
                              });

            return response()->json([
                'unattendedCount' => $unattendedCount,
                'todayReceiptsCount' => $todayReceiptsCount,
                'totalUnvalidatedAmount' => $totalUnvalidatedAmount,
                'sales' => $recentSales
            ]);

        } catch (\Exception $e) {
            \Log::error('Validator Dashboard data error: ' . $e->getMessage());
            
            return response()->json([
                'unattendedCount' => 0,
                'todayReceiptsCount' => 0,
                'totalUnvalidatedAmount' => 0,
                'sales' => []
            ], 500);
        }
    }

    public function getUnattendedCount()
    {
        $user = Auth::user();
        
        if ($user->is_role != 6) {
            return response()->json(['count' => 0]);
        }

        $count = Sale::where('validity', 'unvalidated')->count();

        return response()->json(['count' => $count]);
    }

    /**
     * Alternative method to mark multiple receipts as attended
     */
    public function markMultipleAsAttended(Request $request)
    {
        $user = Auth::user();

        if ($user->is_role != 6) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        $saleIds = $request->input('sale_ids', []);

        try {
            $updatedCount = Sale::whereIn('id', $saleIds)
                               ->where('validity', 'unvalidated')
                               ->update([
                                   'validity' => 'validated',
                                   'verified_by' => $user->id,
                                   'verified_at' => Carbon::now(),
                               ]);

            \Log::info("{$updatedCount} sales validated by user ID {$user->id}");

            return response()->json([
                'success' => true,
                'message' => "{$updatedCount} receipts have been validated successfully",
                'updated_count' => $updatedCount
            ]);

        } catch (\Exception $e) {
            \Log::error('Mark multiple as attended error: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'error' => 'Failed to validate receipts: ' . $e->getMessage()
            ], 500);
        }
    }
}